/**********************************************************************************
 *
 * @file    ald_conf.h
 * @brief   Enable/Disable the peripheral module.
 *
 * @date    3 Aug 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          3 Aug 2021      biyq          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
  
/* Includes ----------------------------------------------------------------- */

#include "nand.h"
#include "FlashParams.h"    /* flash parameters specification */

/* Private Macros ----------------------------------------------------------- */

/* Private Variables--------------------------------------------------------- */
nand_handle_t g_nand_handle;
static ald_ebi_nand_timing_t s_nand_timing;

GPIO_TypeDef * EBI_DATA_PORT[EBI_NAND_DATA_PORTn] = 
{ 
    EBI_D0_PORT, EBI_D1_PORT,
    EBI_D2_PORT, EBI_D3_PORT,
    EBI_D4_PORT, EBI_D5_PORT,
    EBI_D6_PORT, EBI_D7_PORT 
};

const uint16_t EBI_DATA_PIN[EBI_NAND_DATA_PINn] = 
{  
    EBI_D0_PIN, EBI_D1_PIN,
    EBI_D2_PIN, EBI_D3_PIN,
    EBI_D4_PIN, EBI_D5_PIN,
    EBI_D6_PIN, EBI_D7_PIN 
};

                   
/* Public Variables -------------------------------------------------------- */

/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */


/* Private Function---------------------------------------------------------- */

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
void nand_init(void)
{              
    g_nand_handle.instance          = EBI_NAND_DEVICE;
    g_nand_handle.init.bank         = EBI_NAND_BANK2;
    g_nand_handle.init.wait         = EBI_NAND_WAIT_FEATURE_ENABLE;
    g_nand_handle.init.width        = EBI_NAND_MEM_BUS_WIDTH_8;
    g_nand_handle.init.ecc          = EBI_NAND_ECC_ENABLE;
    g_nand_handle.init.size         = EBI_NAND_ECC_PAGE_SIZE_2048BYTE;
    g_nand_handle.init.cle_time     = 0x0;
    g_nand_handle.init.ale_time     = 0x0;
        
    g_nand_handle.config.page_size  = NAND_PAGE_SIZE;
    g_nand_handle.config.spare_size = NAND_SPARE_AREA_SIZE;
    g_nand_handle.config.block_size = NAND_BLOCK_SIZE;
    g_nand_handle.config.block_nbr  = NAND_BLOCK_NBR;
    g_nand_handle.config.plane_nbr  = NAND_PLANE_NBR;
    g_nand_handle.config.plane_size = NAND_PLANE_SIZE;
       
    /*限制条件为：nand flash 手册中的相关参数 + 芯片外设限制。 
    HCLK2最大频率72M，需求<= 13ns的不考虑*/           
    s_nand_timing.hiz_time    = (((ald_cmu_get_hclk2_clock())*7)/50000000);
    s_nand_timing.wait_time   = ((s_nand_timing.hiz_time) * 4) / 3;  
    s_nand_timing.hold_time   = s_nand_timing.wait_time / 2;    
    s_nand_timing.time        = s_nand_timing.hold_time;
    
    if((s_nand_timing.hiz_time - s_nand_timing.time) < 2)
    {
        s_nand_timing.wait_time++;
        s_nand_timing.hold_time++;
    }
    
    /*将实际时序转为寄存器值+时序寄存器范围限制*/                                                                     
    s_nand_timing.hiz_time     = (s_nand_timing.hiz_time < 0x01)?  0x00 : (s_nand_timing.hiz_time - 1);  /*MEMHIZTx/ATTHIZTx   reg范围 0x00--0xFE  HCLK周期 = reg+1 */  
    s_nand_timing.hiz_time     = (s_nand_timing.hiz_time > 0xFE)?  0xFE : s_nand_timing.hiz_time;    
    s_nand_timing.hold_time    = (s_nand_timing.hold_time < 0x01)? 0x01 : s_nand_timing.hold_time;       /*MEMHOLDx/ATTHOLDx   reg范围 0x01--0xFE  HCLK周期 = reg   */  
    s_nand_timing.hold_time    = (s_nand_timing.hold_time > 0xFE)? 0xFE : s_nand_timing.hold_time;                 
    s_nand_timing.wait_time    = (s_nand_timing.wait_time < 0x01)? 0x00 : (s_nand_timing.wait_time - 1); /*MEMWAITx/ATTWAITx   reg范围 0x01--0xFF  HCLK周期 = reg+1 */ 
    s_nand_timing.wait_time    = (s_nand_timing.wait_time > 0xFF)? 0xFF : s_nand_timing.wait_time;  
    s_nand_timing.time         = (s_nand_timing.time < 0x01)?      0x00 : (s_nand_timing.time - 1);      /*MEMSETUPx/ATTSETUPx reg范围 0x00--0xFE  HCLK周期 = reg+1 */
    s_nand_timing.time         = (s_nand_timing.time > 0xFE)?      0xFE : s_nand_timing.time;      
    
    /* NAND Flash initialization */
    ald_nand_init(&g_nand_handle, &s_nand_timing, &s_nand_timing);

    ald_nand_reset(&g_nand_handle);
}

/**
  * @brief  Nand pin init.
  * @param  None
  * @retval None
  */
void nand_pin_init(void)
{
    uint8_t i;
    gpio_init_t gpio_config;

    gpio_config.mode  = GPIO_MODE_OUTPUT;
    gpio_config.odos  = GPIO_PUSH_PULL;
    gpio_config.pupd  = GPIO_PUSH_UP;
    gpio_config.nodrv = GPIO_OUT_DRIVE_0_1;
    gpio_config.podrv = GPIO_OUT_DRIVE_0_1;
    gpio_config.flt   = GPIO_FILTER_DISABLE;
    gpio_config.type  = GPIO_TYPE_TTL;
    gpio_config.func  = GPIO_FUNC_7;

    /* Data pin init */
    for (i = 0; i < EBI_NAND_DATA_PINn; ++i)
    {
        ald_gpio_init(EBI_DATA_PORT[i], EBI_DATA_PIN[i], &gpio_config);
    }

    /* Control pin init */
    gpio_config.mode  = GPIO_MODE_OUTPUT;
    gpio_config.odos  = GPIO_PUSH_PULL;
    gpio_config.pupd  = GPIO_PUSH_UP;
    gpio_config.nodrv = GPIO_OUT_DRIVE_0_1;
    gpio_config.podrv = GPIO_OUT_DRIVE_0_1;
    gpio_config.flt   = GPIO_FILTER_DISABLE;
    gpio_config.type  = GPIO_TYPE_TTL;
    gpio_config.func  = GPIO_FUNC_7;

    /* Output enable pin init */
    ald_gpio_init(EBI_NOE_PORT, EBI_NOE_PIN, &gpio_config);
    /* Write enable init EBI_NWE*/
    ald_gpio_init(EBI_NWE_PORT, EBI_NWE_PIN, &gpio_config);
    /* Chip select pin init EBI_NCE2 */
    ald_gpio_init(EBI_NCE2_PORT, EBI_NCE2_PIN, &gpio_config);
    /* EBI_CLE init */
    ald_gpio_init(EBI_CLE_PORT, EBI_CLE_PIN, &gpio_config);
    /* Address Latch pin init EBI_ALE */
    ald_gpio_init(EBI_ALE_PORT, EBI_ALE_PIN, &gpio_config);
    /* Ready/Busy pin init */
    gpio_config.mode = GPIO_MODE_INPUT;
    gpio_config.func = GPIO_FUNC_7;
    ald_gpio_init(EBI_NWAIT_PORT, EBI_NWAIT_PIN, &gpio_config);
}

/**
  * @brief  Translate logical address into a phy one.
  * @param  addr: Address
  * @param  n_addr: NAND addr.
  * @retval Status
  */
void nand_addr_get(uint32_t addr, nand_address_t *n_addr)
{
    n_addr->page  = (addr % (NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE))) / (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE);
    n_addr->block = (addr % (NAND_PLANE_SIZE * NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE))) / (NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE));
    n_addr->plane = addr / (NAND_PLANE_SIZE * NAND_BLOCK_SIZE * (NAND_PAGE_SIZE + NAND_SPARE_AREA_SIZE));
}

/**
  * @brief  NAND memory Block erase 
  * @param  hperh: pointer to a nand_handle_t structure 
  * @param  addr: pointer to NAND address structure
  * @retval ald status
  */
ald_status_t flm_nand_erase_block(nand_handle_t *hperh, nand_address_t *addr)
{
    uint32_t deviceaddr = 0UL;
    __IO uint32_t timecount = 0;

    __LOCK(hperh);

    if (hperh->state == ALD_NAND_STATE_BUSY)
        return BUSY;

    /* Identify the device address */
    if (hperh->init.bank == EBI_NAND_BANK2)
        deviceaddr = NAND_DEVICE1;
    else
        deviceaddr = NAND_DEVICE2;

    /* Update the NAND controller state */
    hperh->state = ALD_NAND_STATE_BUSY;

    /* Send Erase block command sequence */
    *(__IO uint8_t *)((uint32_t)(deviceaddr | CMD_AREA)) = NAND_CMD_ERASE0;
    *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_1ST_CYCLE(ARRAY_ADDRESS(addr, hperh));
    *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_2ND_CYCLE(ARRAY_ADDRESS(addr, hperh));
    *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_3RD_CYCLE(ARRAY_ADDRESS(addr, hperh));
    *(__IO uint8_t *)((uint32_t)(deviceaddr | CMD_AREA)) = NAND_CMD_ERASE1;

    /* Update the NAND controller state */
    hperh->state = ALD_NAND_STATE_READY;

    /* Read status until NAND is ready */
    while (ald_nand_read_status(hperh) != NAND_READY) {
        if (timecount < 400000){
            timecount++;
        }
        else{
            return TIMEOUT;
        }
    }

    __UNLOCK(hperh);

    return OK;
}

/**
  * @brief  Write page(s) to NAND memory block (8-bits addressing)
  * @param  hperh: pointer to a nand_handle_t structure 
  * @param  addr: pointer to NAND address structure
  * @param  buf: pointer to source buffer to write  
  * @param  nr: number of pages to write to block 
  * @retval ald status
  */
ald_status_t flm_nand_write_page_8b(nand_handle_t *hperh, nand_address_t *addr, uint8_t *buf, uint32_t nr)
{
    __IO uint32_t index = 0;
    __IO uint32_t timecount = 0;
    __IO uint32_t deviceaddr = 0, size = 0, num = 0, nandaddr = 0;

    __LOCK(hperh);

    if (hperh->state == ALD_NAND_STATE_BUSY)
        return BUSY;

    /* Identify the device address */
    if (hperh->init.bank == EBI_NAND_BANK2)
        deviceaddr = NAND_DEVICE1;
    else
        deviceaddr = NAND_DEVICE2;

    /* Update the NAND controller state */ 
    hperh->state = ALD_NAND_STATE_BUSY;

    /* NAND raw address calculation */
    nandaddr = ARRAY_ADDRESS(addr, hperh);

    /* Page(s) write loop */
    while ((nr != 0) && (nandaddr < ((hperh->config.block_size) * (hperh->config.block_nbr)))) {
        /* update the buffer size */
        size = hperh->config.page_size + ((hperh->config.page_size) * num);

        /* Send write page command sequence */
        *(__IO uint8_t *)((uint32_t)(deviceaddr | CMD_AREA)) = NAND_CMD_AREA_A;
        *(__IO uint8_t *)((uint32_t)(deviceaddr | CMD_AREA)) = NAND_CMD_WRITE0;

        /* Cards with page size <= 512 bytes */
        if ((hperh->config.page_size) <= 512U) {
            if (((hperh->config.block_size)* (hperh->config.block_nbr)) <= 65536U) {
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = 0x00;
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_1ST_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_2ND_CYCLE(nandaddr);
            }
            else {/* ((hperh->config.block_size)*(hperh->config.block_nbr)) > 65536 */
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = 0x00;
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_1ST_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_2ND_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_3RD_CYCLE(nandaddr);
            }
        }
        else {/* (hperh->config.page_size) > 512 */
            if (((hperh->config.block_size)* (hperh->config.block_nbr)) <= 65536U) {
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = 0x00;
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = 0x00;
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_1ST_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_2ND_CYCLE(nandaddr);
            }
            else {/* ((hperh->config.block_size)*(hperh->config.block_nbr)) > 65536 */
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = 0x00;
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = 0x00;
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_1ST_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_2ND_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_3RD_CYCLE(nandaddr);
            }
        }

        /* Write data to memory */
        for (; index < size; index++)
            *(__IO uint8_t *)deviceaddr = *(uint8_t *)buf++;

        *(__IO uint8_t *)((uint32_t)(deviceaddr | CMD_AREA)) = NAND_CMD_WRITE_TRUE1;

        /* Read status until NAND is ready */
        while (ald_nand_read_status(hperh) != NAND_READY) {
            if (timecount < 500000){
                timecount++;
            }
            else{
                return TIMEOUT;
            }
        }

        num++;
        nr--;
        nandaddr = (uint32_t)(nandaddr + 1U);
    }

    hperh->state = ALD_NAND_STATE_READY;
    __UNLOCK(hperh);

    return OK;
}

/**
  * @brief  Write spare area(s) to NAND memory (8-bits addressing)
  * @param  hperh: pointer to a nand_handle_t structure 
  * @param  addr: pointer to NAND address structure
  * @param  buf: pointer to source buffer to write  
  * @param  nr: number of spare areas to write to block
  * @retval ald status
  */
ald_status_t flm_nand_write_sparearea_8b(nand_handle_t *hperh, nand_address_t *addr, uint8_t *buf, uint32_t nr)
{
    __IO uint32_t index = 0;
    __IO uint32_t timecount = 0;
    __IO uint32_t deviceaddr = 0, size = 0, num = 0, nandaddr = 0, columnaddr = 0;

    __LOCK(hperh);

    if (hperh->state == ALD_NAND_STATE_BUSY)
        return BUSY;

    /* Identify the device address */
    if (hperh->init.bank == EBI_NAND_BANK2)
        deviceaddr = NAND_DEVICE1;
    else
        deviceaddr = NAND_DEVICE2;

    /* Update the FSMC_NAND controller state */
    hperh->state = ALD_NAND_STATE_BUSY;
    /* Page address calculation */
    nandaddr = ARRAY_ADDRESS(addr, hperh);
    /* Column in page address */
    columnaddr = COLUMN_ADDRESS(hperh);

    /* Spare area(s) write loop */
    while ((nr != 0) && (nandaddr < ((hperh->config.block_size) * (hperh->config.block_nbr)))) {
        /* update the buffer size */
        size = (hperh->config.spare_size) + ((hperh->config.spare_size) * num);

        /* Cards with page size <= 512 bytes */
        if ((hperh->config.page_size) <= 512U) {
            /* Send write Spare area command sequence */
            *(__IO uint8_t *)((uint32_t)(deviceaddr | CMD_AREA)) = NAND_CMD_AREA_C;
            *(__IO uint8_t *)((uint32_t)(deviceaddr | CMD_AREA)) = NAND_CMD_WRITE0;

            if (((hperh->config.block_size)* (hperh->config.block_nbr)) <= 65536U) {
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = 0x00;
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_1ST_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_2ND_CYCLE(nandaddr);
            }
            else {/* ((hperh->config.block_size)*(hperh->config.block_nbr)) > 65536 */
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = 0x00;
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_1ST_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_2ND_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_3RD_CYCLE(nandaddr);
            }
        }
        else {/* (hperh->config.page_size) > 512 */
            /* Send write Spare area command sequence */
            *(__IO uint8_t *)((uint32_t)(deviceaddr | CMD_AREA)) = NAND_CMD_AREA_A;
            *(__IO uint8_t *)((uint32_t)(deviceaddr | CMD_AREA)) = NAND_CMD_WRITE0;

            if (((hperh->config.block_size)* (hperh->config.block_nbr)) <= 65536U) {
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = COLUMN_1ST_CYCLE(columnaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = COLUMN_2ND_CYCLE(columnaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_1ST_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_2ND_CYCLE(nandaddr);
            }
            else {/* ((hperh->config.block_size)*(hperh->config.block_nbr)) > 65536 */
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = COLUMN_1ST_CYCLE(columnaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = COLUMN_2ND_CYCLE(columnaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_1ST_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_2ND_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_3RD_CYCLE(nandaddr);
            }
        }

        /* Write data to memory */
        for (; index < size; index++)
            *(__IO uint8_t *)deviceaddr = *(uint8_t *)buf++;

        *(__IO uint8_t *)((uint32_t)(deviceaddr | CMD_AREA)) = NAND_CMD_WRITE_TRUE1;
        
        /* Read status until NAND is ready */
        while (ald_nand_read_status(hperh) != NAND_READY) {
            if (timecount < 400000){
                timecount++;
            }
            else{
                return TIMEOUT;
            }
        }

        num++;
        nr--;
        nandaddr = (uint32_t)(nandaddr + 1U);
    }

    hperh->state = ALD_NAND_STATE_READY;
    __UNLOCK(hperh);

    return OK;
}

/**
  * @brief  Read page(s) from NAND memory block (8-bits addressing)
  * @param  hperh: pointer to a nand_handle_t structure 
  * @param  addr: pointer to NAND address structure
  * @param  buf: pointer to destination read buffer
  * @param  nr: number of pages to read from block 
  * @retval ald status
  */
ald_status_t flm_nand_read_page_8b(nand_handle_t *hperh, nand_address_t *addr, uint8_t *buf, uint32_t nr)
{
    __IO uint32_t index  = 0;
    __IO uint32_t timecount = 0;
    uint32_t deviceaddr = 0UL, size = 0UL, num = 0UL, nandaddr = 0UL;

    __LOCK(hperh);

    if (hperh->state == ALD_NAND_STATE_BUSY)
        return BUSY;

    /* Identify the device address */
    if (hperh->init.bank == EBI_NAND_BANK2)
        deviceaddr = NAND_DEVICE1;
    else
        deviceaddr = NAND_DEVICE2;

    hperh->state = ALD_NAND_STATE_BUSY;

    /* NAND raw address calculation */
    nandaddr = ARRAY_ADDRESS(addr, hperh);

    /* Page(s) read loop */
    while ((nr != 0) && (nandaddr < ((hperh->config.block_size) * (hperh->config.block_nbr)))) {
        /* update the buffer size */
        size = (hperh->config.page_size) + ((hperh->config.page_size) * num);

        /* Send read page command sequence */
        *(__IO uint8_t *)((uint32_t)(deviceaddr | CMD_AREA)) = NAND_CMD_AREA_A;

        /* Cards with page size <= 512 bytes */
        if ((hperh->config.page_size) <= 512U) {
            if (((hperh->config.block_size)* (hperh->config.block_nbr)) <= 65536U) {
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = 0x00;
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_1ST_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_2ND_CYCLE(nandaddr);
            }
            else { /* ((hperh->config.block_size)*(hperh->config.block_nbr)) > 65536 */
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = 0x00;
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_1ST_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_2ND_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_3RD_CYCLE(nandaddr);
            }
        }
        else { /* (hperh->config.page_size) > 512 */
            if (((hperh->config.block_size)* (hperh->config.block_nbr)) <= 65536U) {
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = 0x00;
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = 0x00;
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_1ST_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_2ND_CYCLE(nandaddr);
            }
            else {/* ((hperh->config.block_size)*(hperh->config.block_nbr)) > 65536 */
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = 0x00;
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = 0x00;
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_1ST_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_2ND_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_3RD_CYCLE(nandaddr);
            }
        }

        *(__IO uint8_t *)((uint32_t)(deviceaddr | CMD_AREA))  = NAND_CMD_AREA_TRUE1;

        /* Check if an extra command is needed for reading pages  */
        if (hperh->config.extra_cmd == ENABLE) {
            /* Read status until NAND is ready */
            while (ald_nand_read_status(hperh) != NAND_READY) {
                if (timecount < 400000){
                    timecount++;
                }
                else{
                    return TIMEOUT;
                }
            }

            /* Go back to read mode */
            *(__IO uint8_t *)((uint32_t)(deviceaddr | CMD_AREA)) = ((uint8_t)0x00);
        }

        /* max 25us tr referred to nand flash manual */
        for (timecount = 0; timecount < 0x100; timecount++)
            timecount++;
            
        /* Get Data into Buffer */    
        for (; index < size; index++) {
            *(uint8_t *)buf++ = *(uint8_t *)deviceaddr;
        }

        num++;
        nr--;
        nandaddr = (uint32_t)(nandaddr + 1U);
    }

    hperh->state = ALD_NAND_STATE_READY;
    __UNLOCK(hperh);

    return OK;
}

/**
  * @brief  Read spare area(s) from NAND memory (8-bits addressing)
  * @param  hperh: pointer to a nand_handle_t structure 
  * @param  addr: pointer to NAND address structure
  * @param  buf: pointer to source buffer to write  
  * @param  nr: Number of spare area to read  
  * @retval ald status
*/
ald_status_t flm_nand_read_sparearea_8b(nand_handle_t *hperh, nand_address_t *addr, uint8_t *buf, uint32_t nr)
{
    __IO uint32_t index = 0;
    __IO uint32_t timecount = 0;
    uint32_t deviceaddr = 0UL, size = 0UL, num = 0UL, nandaddr = 0UL, columnaddr = 0UL;

    /* Process Locked */
    __LOCK(hperh);

    if (hperh->state == ALD_NAND_STATE_BUSY)
        return BUSY;

    /* Identify the device address */
    if (hperh->init.bank == EBI_NAND_BANK2)
        deviceaddr = NAND_DEVICE1;
    else
        deviceaddr = NAND_DEVICE2;

    /* Update the NAND controller state */
    hperh->state = ALD_NAND_STATE_BUSY;
    /* NAND raw address calculation */
    nandaddr = ARRAY_ADDRESS(addr, hperh);
    /* Column in page address */
    columnaddr = COLUMN_ADDRESS(hperh);

    /* Spare area(s) read loop */ 
    while ((nr != 0) && (nandaddr < ((hperh->config.block_size) * (hperh->config.block_nbr)))) {
        /* update the buffer size */
        size = (hperh->config.spare_size) + ((hperh->config.spare_size) * num);

        /* Cards with page size <= 512 bytes */
        if ((hperh->config.page_size) <= 512U) {
            /* Send read spare area command sequence */     
            *(__IO uint8_t *)((uint32_t)(deviceaddr | CMD_AREA)) = NAND_CMD_AREA_C;

            if (((hperh->config.block_size)* (hperh->config.block_nbr)) <= 65536U) {
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = 0x00;
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_1ST_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_2ND_CYCLE(nandaddr);
            }
            else {/* ((hperh->config.block_size)*(hperh->config.block_nbr)) > 65536 */
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = 0x00;
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_1ST_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_2ND_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_3RD_CYCLE(nandaddr);
            }
        }
        else {/* (hperh->config.page_size) > 512 */
            /* Send read spare area command sequence */ 
            *(__IO uint8_t *)((uint32_t)(deviceaddr | CMD_AREA)) = NAND_CMD_AREA_A;

            if (((hperh->config.block_size)* (hperh->config.block_nbr)) <= 65536U) {
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = COLUMN_1ST_CYCLE(columnaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = COLUMN_2ND_CYCLE(columnaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_1ST_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_2ND_CYCLE(nandaddr);
            }
            else {/* ((hperh->config.block_size)*(hperh->config.block_nbr)) > 65536 */
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = COLUMN_1ST_CYCLE(columnaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = COLUMN_2ND_CYCLE(columnaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_1ST_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_2ND_CYCLE(nandaddr);
                *(__IO uint8_t *)((uint32_t)(deviceaddr | ADDR_AREA)) = ADDR_3RD_CYCLE(nandaddr);
            }
        }

        *(__IO uint8_t *)((uint32_t)(deviceaddr | CMD_AREA)) = NAND_CMD_AREA_TRUE1;

        if (hperh->config.extra_cmd == ENABLE) {
            
            /* Read status until NAND is ready */
            while (ald_nand_read_status(hperh) != NAND_READY) {
                if (timecount < 400000){
                    timecount++;
                }
                else{
                    return TIMEOUT;
                }
            }

            /* Go back to read mode */
            *(__IO uint8_t *)((uint32_t)(deviceaddr | CMD_AREA)) = ((uint8_t)0x00);
        }

        for (timecount = 0; timecount < 0x100; timecount++)
            timecount++;
        
        /* Get Data into Buffer */
        for (; index < size; index++)
            *(uint8_t *)buf++ = *(uint8_t *)deviceaddr;

        num++;
        nr--;
        nandaddr = (uint32_t)(nandaddr + 1U);
    }

    hperh->state = ALD_NAND_STATE_READY;
    __UNLOCK(hperh);

    return OK;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
